/**************************************************************************************
 
   Copyright (c) Hilscher GmbH. All Rights Reserved.
 
 **************************************************************************************
 
   Filename:
    $Id: PhysicalInterface.cpp 3138 2011-11-29 16:44:42Z stephans $
   Last Modification:
    $Author: stephans $
    $Date: 2011-11-29 17:44:42 +0100 (Di, 29 Nov 2011) $
    $Revision: 3138 $
   
   Targets:
     Win32/ANSI   : yes
     Win32/Unicode: yes (define _UNICODE)
     WinCE        : no
 
   Description:
    Implementation of the physical interface class
       
   Changes:
 
     Version   Date        Author   Description
     ----------------------------------------------------------------------------------
     1         xx                   created
 
**************************************************************************************/

/////////////////////////////////////////////////////////////////////////////
/// \file PhysicalInterface.cpp
/// Implementation of the physical interface class
/////////////////////////////////////////////////////////////////////////////

#include "StdAfx.h"

#include "HilTransport.h"
#include "PhysicalInterface.h" 
#include "TransportLayer.h"
#include "DeviceHandler.h"  

/////////////////////////////////////////////////////////////////////////////
/// Callback for notifications 
///   \param    szInterface   Name of the interface
///   \param    eNotify       Notifify value
///   \param    pvLayer       Reference of the layer 
///   \param    pvUser        User-Pointer 
/////////////////////////////////////////////////////////////////////////////
static void fInterfaceNotifyCallback( const char*                   szInterface,
                                      NETX_INTERFACE_NOTIFICATION_E eNotify,
                                      void*                         pvLayer, 
                                      void*                         pvUser)
{
  UNREFERENCED_PARAMETER(pvLayer);

  CPhysicalLayer* pcLayer = reinterpret_cast<CPhysicalLayer*>(pvUser);

  std::string cInterface(szInterface);

  pcLayer->m_pcDeviceHandler->HandleInterfaceNotification( cInterface, eNotify, pcLayer);
}

/////////////////////////////////////////////////////////////////////////////
/// Initialize physical layer (open the connector)
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CPhysicalLayer::Init(void)
{ 
  return m_ptConnector->tFncTable.pfnNetXConOpen( fInterfaceNotifyCallback, this);
}

/////////////////////////////////////////////////////////////////////////////
/// DeInitialize physical layer (close the connector)
/////////////////////////////////////////////////////////////////////////////
void CPhysicalLayer::Deinit(void)
{ 
  m_ptConnector->tFncTable.pfnNetXConClose();
}

/////////////////////////////////////////////////////////////////////////////
/// Create a dialog in the given window (hDiag)
///   \param    hDiag   Reference of the dialog window 
///   \return   CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
HWND CPhysicalLayer::CreateConfigDialog( HWND hParentWnd, const char* szConfig)
{        
  return m_ptConnector->tFncTable.pfnNetXConCreateDialog( hParentWnd, szConfig);
}

/////////////////////////////////////////////////////////////////////////////
/// Close the given configuration dialog 
///   \return   CIFX_NO_ERROR  on success
/////////////////////////////////////////////////////////////////////////////
void CPhysicalLayer::EndDialog( void )
{
  m_ptConnector->tFncTable.pfnNetXConEndDialog();
}

/////////////////////////////////////////////////////////////////////////////
/// Get configuration of connector
///   \return   CIFX_NO_ERROR  on success
/////////////////////////////////////////////////////////////////////////////
int32_t CPhysicalLayer::GetConfig( NETX_CONNECTOR_CONFIG_CMD_E eCmd, void* pvConfig)
{ 
  if(NULL == pvConfig)
    return CIFX_INVALID_POINTER;

  return m_ptConnector->tFncTable.pfnNetXConGetConfig( eCmd, pvConfig);         
}

/////////////////////////////////////////////////////////////////////////////
/// Set configuration of connector 
///   \param  eCmd      Set configuration command
///   \param  szConfig  Configuration string
///   \return   CIFX_NO_ERROR  on success
/////////////////////////////////////////////////////////////////////////////
int32_t CPhysicalLayer::SetConfig( NETX_CONNECTOR_CONFIG_CMD_E eCmd, const char* szConfig)
{                                     
  if(NULL == szConfig)
    return CIFX_INVALID_POINTER;

  return m_ptConnector->tFncTable.pfnNetXConSetConfig( eCmd, szConfig);         
}

/////////////////////////////////////////////////////////////////////////////
/// Get type of connector
///   \return Type of connector (see NETX_CONNECTOR_TYPE_U)
/////////////////////////////////////////////////////////////////////////////
uint32_t CPhysicalLayer::GetType( void)
{
  uint32_t ulType = 0;
  m_ptConnector->tFncTable.pfnNetXConGetInformation( eTYPE,
                                                     sizeof(ulType), 
                                                     &ulType);
  return ulType;
}

/////////////////////////////////////////////////////////////////////////////
/// Get short interface name 
///   \param  ulSize       Size of buffer referenced by pszShortName
///   \param  pszShortName Buffer return short interface name
///   \param  pszLongName  Corresponding long interface name
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CPhysicalLayer::GetShortInterfaceName( uint32_t ulSize, char* pszShortName, char* pszLongName)
{
  INTERFACE_NAME_TRANSLATION_T  tTranslate         = {0};

  tTranslate.szSource          = pszLongName;
  tTranslate.szDestination     = pszShortName;
  tTranslate.ulDestinationSize = ulSize;

  return m_ptConnector->tFncTable.pfnNetXConGetInformation( eSHORT_INTERFACE_NAME,
                                                            sizeof(tTranslate), 
                                                            &tTranslate);
}

/////////////////////////////////////////////////////////////////////////////
/// Get short interface name 
///   \param  ulSize       Size of buffer referenced by pszLongName
///   \param  pszLongName  Buffer return long interface name
///   \param  pszShortName Corresponding short interface name
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CPhysicalLayer::GetLongInterfaceName( uint32_t ulSize, char* pszLongName, char* pszShortName)
{
  INTERFACE_NAME_TRANSLATION_T  tTranslate         = {0};

  tTranslate.szSource          = pszShortName;
  tTranslate.szDestination     = pszLongName;
  tTranslate.ulDestinationSize = ulSize;

  return m_ptConnector->tFncTable.pfnNetXConGetInformation( eLONG_INTERFACE_NAME,
                                                            sizeof(tTranslate), 
                                                            &tTranslate);
}

/////////////////////////////////////////////////////////////////////////////
/// Callback for receiving data 
///   \param    pabRxData     Reference of received data 
///   \param    ulRXDataLen   Length of received data
///   \param    pvUser        User-Pointer 
/////////////////////////////////////////////////////////////////////////////
static void fReceiveDataCallback( unsigned char*  pabRxData,
                                    unsigned long   ulRXDataLen,
                                    void*           pvUser)
{
  CPhysicalInterface* pcPhysIF    = reinterpret_cast<CPhysicalInterface*>(pvUser);
  CTransportLayer*    pcTransport = pcPhysIF->GetTransportLayer();

  pcTransport->ReceiveData( pabRxData, ulRXDataLen);
}

/////////////////////////////////////////////////////////////////////////////
/// Start physical layer (connector)
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CPhysicalInterface::Start( void)
{
  return m_ptConnector->tFncTable.pfnNetXConIntfStart(m_pvConnectorInterface,
                                                      fReceiveDataCallback, 
                                                      this);
}

/////////////////////////////////////////////////////////////////////////////
/// Get state of the interface
///   \return   State of the interface
/////////////////////////////////////////////////////////////////////////////
uint32_t CPhysicalInterface::GetState( void)
{
  uint32_t ulState = eINTERFACE_STATE_NOT_SET;

  m_ptConnector->tFncTable.pfnNetXConIntfGetInformation( m_pvConnectorInterface,
                                                         eINTERFACE_STATE,
                                                         sizeof(ulState),
                                                         &ulState);
  return ulState;
}

/////////////////////////////////////////////////////////////////////////////
/// Get send timeout of the interface
///   \return   Send timeout of the interface
/////////////////////////////////////////////////////////////////////////////
uint32_t CPhysicalInterface::GetSendTimeout( void)
{
  uint32_t ulTimeout = 0;

  m_ptConnector->tFncTable.pfnNetXConIntfGetInformation( m_pvConnectorInterface,
                                                         eSEND_TIMEOUT,
                                                         sizeof(ulTimeout),
                                                         &ulTimeout);
  return ulTimeout;
}

/////////////////////////////////////////////////////////////////////////////
/// Get keep alive timeout of the interface
///   \return   Keep alive timeout of the interface
/////////////////////////////////////////////////////////////////////////////
uint32_t CPhysicalInterface::GetKeepAliveTimeout( void)
{
  uint32_t ulTimeout = HIL_TRANSPORT_KEEP_ALIVE_CLIENT_TIMEOUT;

  m_ptConnector->tFncTable.pfnNetXConIntfGetInformation( m_pvConnectorInterface,
                                                         eKEEPALIVE_TIMEOUT,
                                                         sizeof(ulTimeout),
                                                         &ulTimeout);
  return ulTimeout;
}

/////////////////////////////////////////////////////////////////////////////
/// Get reset timeout of the interface
///   \return   Reset timeout of the interface
/////////////////////////////////////////////////////////////////////////////
uint32_t CPhysicalInterface::GetResetTimeout( void)
{
  uint32_t ulTimeout = 0;

  m_ptConnector->tFncTable.pfnNetXConIntfGetInformation( m_pvConnectorInterface,
                                                         eRESET_TIMEOUT,
                                                         sizeof(ulTimeout),
                                                         &ulTimeout);
  return ulTimeout;
}